#!/bin/sh
#
# Rootkit Detection Scanner v1.0-rev1 - POSIX port (no $(), no mktemp)
#   - Ported from rootkit_detect_scanner_v1.0.sh (bash)
#   - Uses strict POSIX /bin/sh syntax (no $(), no $(( )) )
#
set -u

#############################################
# PATH update for Solaris (xpg4 utilities)
#############################################
if [ -d /usr/xpg4/bin ]; then
    case ":$PATH:" in
        *":/usr/xpg4/bin:"*) : ;;
        *) PATH="/usr/xpg4/bin:$PATH" ;;
    esac
fi
export PATH

#############################################
# Colors (fallback if tput not available)
#############################################
if command -v tput >/dev/null 2>&1; then
    RED=`tput setaf 1`
    GREEN=`tput setaf 2`
    YELLOW=`tput setaf 3`
    BLUE=`tput setaf 4`
    RESET=`tput sgr0`
else
    RED=""
    GREEN=""
    YELLOW=""
    BLUE=""
    RESET=""
fi

#############################################
# Global flags and lists
#############################################
HIDDEN_FOUND=0
ROOTKIT_FOUND=0
BACKDOOR_FOUND=0

HIDDEN_LIST=""
ROOTKIT_LIST=""
BACKDOOR_LIST=""

ROOTKIT_NAME=""
BACKDOOR_PATH=""
MODULE_NAME=""
PROC_ENTRY=""

VISIBLE_FILE_ANALYSIS=0

#############################################
# Utility functions
#############################################
have_cmd() {
    command -v "$1" >/dev/null 2>&1
}

# Simple tmp file generator (replaces mktemp)
make_tmp() {
    base="$1"
    ts=`date +%s 2>/dev/null | tr -d '\n'`
    if [ -z "$ts" ]; then
        ts="0"
    fi
    echo "/tmp/${base}.$$.$ts"
}

get_ip_addr() {
    out=""
    # 1) ip -4 addr
    if command -v ip >/dev/null 2>&1; then
        out=`ip -4 addr show 2>/dev/null | awk '/ inet / && $2 !~ /^127\./ { split($2,a,"/"); print a[1]; exit }'`
        [ -n "$out" ] && { echo "$out"; return; }
    fi
    # 2) ifconfig (plain)
    if command -v ifconfig >/dev/null 2>&1; then
        out=`ifconfig 2>/dev/null \
           | sed -n 's/.*inet \([0-9][0-9]*\.[0-9][0-9]*\.[0-9][0-9]*\.[0-9][0-9]*\).*/\1/p' \
           | grep -v '^127\.' | awk 'NR==1{print; exit}'`
        [ -n "$out" ] && { echo "$out"; return; }

        # 3-1) ifconfig -a (AIX/Solaris/HP-UX)
        out=`ifconfig -a 2>/dev/null | awk '
            /inet / && $0 !~ /127\.0\.0\.1/ && $0 !~ /inet6/ {
                for (i=1;i<=NF;i++) {
                    if ($i=="inet")      { ip=$(i+1); gsub("addr:","",ip);
                        if (ip ~ /^[0-9]+(\.[0-9]+){3}$/) { print ip; exit } }
                    else if ($i ~ /^inet:/) { ip=$i; sub("inet:","",ip); sub("addr:","",ip);
                        if (ip ~ /^[0-9]+(\.[0-9]+){3}$/ && ip !~ /^127\./) { print ip; exit } }
                }
            }'`
        [ -n "$out" ] && { echo "$out"; return; }

        # 3-2) sed fallback
        out=`ifconfig -a 2>/dev/null \
           | sed -n 's/.*inet \([0-9][0-9]*\.[0-9][0-9]*\.[0-9][0-9]*\.[0-9][0-9]*\).*/\1/p' \
           | grep -v '^127\.' | awk 'NR==1{print; exit}'`
        [ -n "$out" ] && { echo "$out"; return; }
    fi
    # 3) getent
    if command -v getent >/dev/null 2>&1; then
        out=`getent ahostsv4 "$HOSTNAME" 2>/dev/null | awk '$1 !~ /^127\./ {print $1; exit}'`
        [ -n "$out" ] && { echo "$out"; return; }
    fi
    echo unknown_ip
}

HOSTNAME_STR=`hostname 2>/dev/null || echo "unknown_host"`
IP_STR=`get_ip_addr`
TIME_STR=`date '+%Y%m%d_%H%M%S' 2>/dev/null || date +%s`
LOG_FILE="${HOSTNAME_STR}_${IP_STR}_result_${TIME_STR}.log"

log() {
    msg=$1
    # stderr (with color)
    printf '%s\n' "$msg" >&2
    # logfile without ANSI
    printf '%s\n' "$msg" | sed -e 's/\x1b\[[0-9;]*[a-zA-Z]//g' -e 's/\x1b(B//g' >>"$LOG_FILE" 2>/dev/null || :
}

need_root() {
    if have_cmd id; then
        uid=`id -u 2>/dev/null || echo 1`
        if [ "$uid" -ne 0 ]; then
            printf '%s\n' "${RED}[ERROR]${RESET} Run as root(uid=0)" >&2
            exit 1
        fi
        log "${GREEN}[OK]${RESET} Running with root privileges (uid=0)"
    else
        log "${GREEN}[PASS]${RESET} uid command not found!"
    fi
}


get_system_info() {
    kernel_version=`uname -r 2>/dev/null || echo "Unknown"`
    machine_arch=`uname -m 2>/dev/null || echo "Unknown"`
    os_release=""

    if [ -f /etc/os-release ]; then
        os_release=`grep "^PRETTY_NAME=" /etc/os-release 2>/dev/null | cut -d'"' -f2`
    elif [ -f /etc/redhat-release ]; then
        os_release=`cat /etc/redhat-release 2>/dev/null`
    elif [ -f /etc/debian_version ]; then
        os_release=`printf 'Debian %s' "\`cat /etc/debian_version 2>/dev/null\`"`
    else
        os_release="Unknown"
    fi

    log "${BLUE}============================================================${RESET}"
    log "${BLUE}[+] System Information${RESET}"
    log "${BLUE}============================================================${RESET}"
    log " - Hostname: $HOSTNAME_STR"
    log " - IP: $IP_STR"
    log " - Kernel: $kernel_version"
    log " - OS: $os_release $machine_arch"
}

calc_sha256() {
    f=$1
    [ -r "$f" ] || { echo ""; return; }
    if have_cmd sha256sum; then
        sha256sum "$f" 2>/dev/null | awk '{print $1}'
    elif have_cmd digest; then
        digest -a sha256 "$f" 2>/dev/null | awk '{print $NF}'
    else
        echo ""
    fi
}

calc_md5() {
    f=$1
    [ -r "$f" ] || { echo ""; return; }
    if have_cmd md5sum; then
        md5sum "$f" 2>/dev/null | awk '{print $1}'
    elif have_cmd digest; then
        digest -a md5 "$f" 2>/dev/null | awk '{print $NF}'
    else
        echo ""
    fi
}

file_details() {
    path=$1

    if [ ! -f "$path" ] && [ ! -d "$path" ]; then
        log " - ${YELLOW}File Not Found: $path${RESET}"
        return
    fi

    mtime=`stat -c '%y' "$path" 2>/dev/null || stat -f '%Sm' "$path" 2>/dev/null || echo "N/A"`
    chtime=`stat -c '%z' "$path" 2>/dev/null || echo "N/A"`
    md5=`calc_md5 "$path"`
    sha256=`calc_sha256 "$path"`

    [ -z "$md5" ] && md5="N/A"
    [ -z "$sha256" ] && sha256="N/A"

    log " - Path: $path"
    log " - Modified: $mtime"
    log " - Changed: $chtime"
    log " - MD5: $md5"
    log " - SHA256: $sha256"
}

#############################################
# List helpers (newline-separated "arrays")
#############################################
append_unique() {
    list=$1
    value=$2

    if [ -z "$list" ]; then
        printf '%s\n' "$value"
        return
    fi

    case "
$list
" in
        *"
$value
"*) printf '%s\n' "$list" ;;
        *) printf '%s\n%s\n' "$list" "$value" ;;
    esac
}

report_hidden() {
    path=$1
    HIDDEN_LIST=`append_unique "$HIDDEN_LIST" "$path"`
    log "  ${RED}[!]${RESET} Hidden Entry File: ${RED}$path${RESET}"
}

report_deleted() {
    path=$1
    log "  ${YELLOW}[!]${RESET} Deleted File: ${YELLOW}$path${RESET}"
}

#############################################
# diff_list: FS list vs ls list
#############################################
diff_list() {
    list1=$1
    list2=$2

    tmp1=`make_tmp rootkit_fs1`
    tmp2=`make_tmp rootkit_fs2`

    printf '%s\n' "$list1" | sort -u >"$tmp1"
    printf '%s\n' "$list2" | sort -u >"$tmp2"

    comm -23 "$tmp1" "$tmp2" 2>/dev/null || :

    rm -f "$tmp1" "$tmp2"
}

#############################################
# Rootkit & backdoor detection
#############################################
find_rootkit_module() {
    full_path=$1
    file_type=$2

    if [ ! -f "$full_path" ]; then
        return
    fi

    # collect insmod targets
    tmp_ins=`make_tmp rootkit_ins`
    strings "$full_path" 2>/dev/null \
        | grep -E "insmod[[:space:]]+" 2>/dev/null \
        | sed -n 's/.*insmod[[:space:]]\+\([^ ]\+\).*/\1/p' >"$tmp_ins"

    if [ ! -s "$tmp_ins" ]; then
        rm -f "$tmp_ins"
        return
    fi

    while IFS= read -r target; do
        [ -z "$target" ] && continue
        case "$target" in
            /*) : ;;
            *) target=`dirname "$full_path"`"$target" ;;
        esac

        ROOTKIT_FOUND=1
        ROOTKIT_LIST=`append_unique "$ROOTKIT_LIST" "$target"`

        if [ "$file_type" = "hidden" ]; then
            log "  [+] Found insmod in hidden entry file: ${RED}$full_path${RESET}"
            log "    ${RED}[!]${RESET} Suspicious Rootkit Found : ${RED}$target${RESET}"
        else
            log "  [+] Found insmod in normal entry file: ${YELLOW}$full_path${RESET}"
            log "    ${RED}[!]${RESET} Suspicious Rootkit Found : ${RED}$target${RESET}"
        fi

        # search candidate backdoor path strings inside module
        tmp_str=`make_tmp rootkit_str`
        strings "$target" 2>/dev/null >"$tmp_str"

        if [ -s "$tmp_str" ]; then
            while IFS= read -r line; do
                [ -z "$line" ] && continue
                # candidate like "//usr/local/bin/..." or similar absolute path
                case "$line" in
                    *//*)
                        cand=`printf '%s\n' "$line" | awk '{$1=$1;print}'`
                        BACKDOOR_FOUND=1
                        BACKDOOR_LIST=`append_unique "$BACKDOOR_LIST" "$cand"`
                        log "    ${RED}[!]${RESET} Suspicious Backdoor Found: ${RED}$cand ${RESET}"
                        ;;
                esac
            done <"$tmp_str"
        fi

        rm -f "$tmp_str"
    done <"$tmp_ins"

    rm -f "$tmp_ins"
}

normalize_fs_list() {
    a=`printf '%s\n' "$1" | sed 's/\\\\x5c/\\\\/g'`
    if [ "$a" != "$1" ]; then
        printf '%s\n' "$a"
    else
        printf '%s\n' "$1" | sed 's/\\\\x/\\x/g'
    fi
}

#############################################
# Filesystem scan (XFS/EXT2/3/4/BTRFS)
#############################################
scan_fs() {
    log "${BLUE}============================================================${RESET}"
    log "${BLUE}[+] Detecting Suspicious file (scan_fs)${RESET}"
    log "${BLUE}============================================================${RESET}"

    for TARGET_PATH in /etc/systemd/system/ /etc/init.d/ /etc/rc2.d/ /etc/rc3.d/ /etc/rc5.d/
    do
        if [ ! -d "$TARGET_PATH" ]; then
            log "[!] Target directory '$TARGET_PATH' does not exist. Skipping scan."
            continue
        fi

        # FS and FSTYPE
        if df -T "$TARGET_PATH" >/dev/null 2>&1; then
            FS=`df -T "$TARGET_PATH" 2>/dev/null | awk 'NR==2 {print $1}'`
            FSTYPE=`df -T "$TARGET_PATH" 2>/dev/null | awk 'NR==2 {print $2}'`
        else
            FS=`df "$TARGET_PATH" 2>/dev/null | awk 'NR==2 {print $1}'`
            FSTYPE=`df -n "$TARGET_PATH" 2>/dev/null | awk 'NR==1 {print $NF}'`
        fi

        TARGET_INODE=`stat -c %i "$TARGET_PATH" 2>/dev/null`
        if [ -z "$TARGET_INODE" ]; then
            TARGET_INODE=`ls -id "$TARGET_PATH" 2>/dev/null | awk '{print $1}'`
        fi

        LS_LIST=`ls -a "$TARGET_PATH" 2>/dev/null | sort`
        log "[*] DIR: $TARGET_PATH, FS=$FS, FSTYPE=$FSTYPE, TARGET_INODE=$TARGET_INODE"

        FS_LIST=""
        case "$FSTYPE" in
            xfs)
                if ! have_cmd xfs_db; then
                    log "[!] xfs_db command not found. Cannot scan XFS filesystem at $TARGET_PATH."
                    continue
                fi
                DBLOCKS=`xfs_db -r "$FS" -c "inode $TARGET_INODE" -c "print" 2>/dev/null | awk '/nblocks/ {print $3}'`
                TMP_XFS=`make_tmp xfs_names`

                if [ -z "$DBLOCKS" ] || [ "$DBLOCKS" -eq 0 ] 2>/dev/null; then
                    xfs_db -r "$FS" -c "inode $TARGET_INODE" -c "print" 2>/dev/null \
                        | awk -F'"' '/name/ {print $2}' | grep -v '^$' | sort >"$TMP_XFS"
                else
                    # read each data block
                    i=0
                    while [ "$i" -lt "$DBLOCKS" ] 2>/dev/null; do
                        xfs_db -r "$FS" -c "inode $TARGET_INODE" -c "dblock $i" -c "print" 2>/dev/null \
                            | awk -F'"' '/name/ {print $2}' >>"$TMP_XFS" 2>/dev/null
                        i=`expr "$i" + 1`
                    done
                fi

                if [ -f "$TMP_XFS" ]; then
                    FS_LIST=`sort -u "$TMP_XFS" | grep -v '^$'`
                    rm -f "$TMP_XFS"
                fi
                ;;
            ext2|ext3|ext4)
                if ! have_cmd debugfs; then
                    log "[!] debugfs command not found. Cannot scan EXT2/3/4 filesystem at $TARGET_PATH."
                    continue
                fi
                FS_LIST=`debugfs -R "ls -l <$TARGET_INODE>" "$FS" 2>/dev/null \
                    | awk 'NF>=8 {print $NF}' \
                    | grep -v '^$' \
                    | sort`
                ;;
            btrfs)
                if ! have_cmd btrfs; then
                    log "[!] btrfs command not found. Cannot scan BTRFS filesystem at $TARGET_PATH."
                    continue
                fi
                FS_LIST=`btrfs inspect-internal dump-tree "$FS" 2>/dev/null \
                    | awk -v inode="$TARGET_INODE" '
                        /DIR_ITEM/ && $0 ~ ("key \\("inode" DIR_ITEM") {show=1}
                        show && /location key/ {is_subvol=($0 ~ /ROOT_ITEM/) }
                        show && /name:/ {
                            sub(/.*name: /,"")
                            name=$0
                            if (!is_subvol) {print name}
                            show=0; is_subvol=0
                        }' \
                    | grep -v '^$' \
                    | sort`
                ;;
            *)
                log "[!] Filesystem type is '$FSTYPE'. Only XFS, EXT2/3/4, and BTRFS are supported for $TARGET_PATH."
                continue
                ;;
        esac
        FS_LIST_NORMALIZED=`normalize_fs_list "$FS_LIST"`
        LS_LIST_NORMALIZED=`printf '%s\n' "$LS_LIST"`

        # diff between FS_LIST and LS_LIST
        hidden_files=`diff_list "$FS_LIST_NORMALIZED" "$LS_LIST_NORMALIZED"`

        if [ -n "$hidden_files" ]; then
            tmp_hidden=`make_tmp rootkit_hidden`
            printf '%s\n' "$hidden_files" >"$tmp_hidden"
            while IFS= read -r file; do
                [ -z "$file" ] && continue
                case "$file" in
                    .|..|selinux) continue ;;
                esac
                full_path=$TARGET_PATH$file
                if [ -f "$full_path" ] || [ -d "$full_path" ]; then
                    HIDDEN_FOUND=1
                    report_hidden "$full_path"
                    if [ -f "$full_path" ]; then
                        find_rootkit_module "$full_path" "hidden"
                    fi
                else
                    report_deleted "$full_path"
                fi
            done <"$tmp_hidden"
            rm -f "$tmp_hidden"
        fi

        if [ "$VISIBLE_FILE_ANALYSIS" -eq 1 ]; then
            visible_files=$LS_LIST
            if [ -n "$visible_files" ]; then
                tmp_vis=`make_tmp rootkit_vis`
                printf '%s\n' "$visible_files" >"$tmp_vis"
                while IFS= read -r file; do
                    [ -z "$file" ] && continue
                    case "$file" in
                        .|..|selinux) continue ;;
                    esac
                    full_path=$TARGET_PATH$file
                    if [ -f "$full_path" ]; then
                        find_rootkit_module "$full_path" "normal"
                    fi
                done <"$tmp_vis"
                rm -f "$tmp_vis"
            fi
        fi
    done
}

#############################################
# Low-level ELF helpers
#############################################
find_section_offset() {
    section=$1
    section_offset=""

    section_info=`readelf -S "$ROOTKIT_NAME" 2>/dev/null | awk -v s="$section" '$0 ~ s {print; exit}'`
    if [ -n "$section_info" ]; then
        section_offset=`echo "$section_info" | awk '{print $5}'`
    fi

    if [ -z "$section_offset" ] || [ "$section_offset" = "00000000" ] || [ "$section_offset" = "0000000000000000" ]; then
        section_info=`readelf -S "$ROOTKIT_NAME" 2>/dev/null | awk '/rodata/ {print; exit}'`
        if [ -n "$section_info" ]; then
            section_offset=`echo "$section_info" | awk '{print $5}'`
        fi
    fi

    printf '%s\n' "$section_offset"
}

extract_string_from_offset() {
    section=$1
    offset=$2
    count=${3:-200}

    section_offset=`find_section_offset "$section"`
    if [ -n "$section_offset" ] && [ "$section_offset" != "0" ]; then
        section_dec=`printf '%d' 0x"$section_offset" 2>/dev/null`
        if [ -n "$offset" ]; then
            off_dec=`printf '%d' 0x"$offset" 2>/dev/null`
            abs_skip=`expr "$section_dec" + "$off_dec"`
        else
            abs_skip="$section_dec"
        fi
        dd if="$ROOTKIT_NAME" bs=1 skip="$abs_skip" count="$count" 2>/dev/null | strings | head -n1
    fi
}

extract_vermagic_info() {
    vermagic_info=""
    if have_cmd strings; then
        vermagic_info=`strings "$ROOTKIT_NAME" 2>/dev/null | grep "vermagic=" | head -n1 | sed 's/vermagic=//'`
    fi
    if [ -n "$vermagic_info" ]; then
        log " - Module Version Magic: $vermagic_info"
    else
        log " - Module Version Magic: ${YELLOW}Not found${RESET}"
    fi
}

analyze_function_calls() {
    func_name=$1
    call_locations=`objdump -drw "$ROOTKIT_NAME" 2>/dev/null | grep -n "$func_name"`

    if [ -z "$call_locations" ]; then
        log "${YELLOW}[info]${RESET} No $func_name calls"
        return
    fi

    tmp_calls=`make_tmp rootkit_calls`
    printf '%s\n' "$call_locations" >"$tmp_calls"

    while IFS= read -r call_line; do
        [ -z "$call_line" ] && continue
        line_number=`printf '%s\n' "$call_line" | cut -d: -f1`
        case "$line_number" in
            *[!0-9]*) continue ;;
        esac

        start_line=`expr "$line_number" - 20`
        if [ "$start_line" -lt 1 ]; then
            start_line=1
        fi
        end_line=`expr "$line_number" + 5`

        call_context=`objdump -drw "$ROOTKIT_NAME" 2>/dev/null | sed -n "${start_line},${end_line}p"`
        rodata_refs=`printf '%s\n' "$call_context" | grep ".rodata" | sort -u`

        [ -z "$rodata_refs" ] && continue

        processed=""
        tmp_refs=`make_tmp rootkit_refs`
        printf '%s\n' "$rodata_refs" >"$tmp_refs"

        while IFS= read -r ref_line; do
            [ -z "$ref_line" ] && continue
            section=`printf '%s\n' "$ref_line" | sed -n 's/.*\(\.rodata[^+ ]*\).*/\1/p'`
            offset=`printf '%s\n' "$ref_line" | sed -n 's/.*+0x\([0-9a-fA-F][0-9a-fA-F]*\).*/\1/p'`

            [ -z "$section" ] && continue
            key="${section}+${offset}"

            case " $processed " in
                *" $key "*) continue ;;
            esac
            processed="$processed $key"

            extracted_string=`extract_string_from_offset "$section" "$offset"`
            [ -z "$extracted_string" ] && continue

            if [ "$func_name" = "call_usermodehelper" ]; then
                case "$extracted_string" in
                    /*)
                        if [ "$extracted_string" != "/bin/sh" ] && [ "$extracted_string" != "/bin/bash" ]; then
                            BACKDOOR_PATH="$extracted_string"
                            BACKDOOR_FOUND=1
                        fi
                        ;;
                esac
            elif [ "$func_name" = "proc_create" ] || [ "$func_name" = "create_proc_entry" ]; then
                echo "extracted_string: $extracted_string"
                if [ -z "$PROC_ENTRY" ]; then
                    case "$extracted_string" in
                        */*)
                            PROC_ENTRY="$extracted_string"
                            ;;
                    esac
                fi
            fi
        done <"$tmp_refs"
        rm -f "$tmp_refs"
    done <"$tmp_calls"

    rm -f "$tmp_calls"
}

extract_module_info() {
    gnu_section_offset=`readelf -S "$ROOTKIT_NAME" 2>/dev/null | awk '/gnu.*linkonce/ {print $5; exit}'`
    if [ -n "$gnu_section_offset" ] && [ "$gnu_section_offset" != "00000000" ]; then
        offset_dec=`printf '%d' 0x"$gnu_section_offset" 2>/dev/null`
        module_name=`dd if="$ROOTKIT_NAME" bs=1 skip="$offset_dec" count=64 2>/dev/null | strings | head -n1`
        if [ -n "$module_name" ]; then
            MODULE_NAME="$module_name"
        fi
    fi
}

p_rootkit_info() {
    if [ ! -f "$ROOTKIT_NAME" ]; then
        log " - ${YELLOW}File Not Found!${RESET}"
        return
    fi

    if ! have_cmd objdump || ! have_cmd readelf || ! have_cmd dd; then
        log "${YELLOW}[!][SKIP] Commands for analysis do not exist.${RESET}"
        return
    fi

    MODULE_NAME=""
    PROC_ENTRY=""
    BACKDOOR_PATH=""

    extract_module_info
    if [ -z "$MODULE_NAME" ]; then
        log "${YELLOW}[!] Unable to extract module name. Skipping further analysis.${RESET}"
        return
    fi

    analyze_function_calls "proc_create"
    if [ -z "$PROC_ENTRY" ]; then
        analyze_function_calls "create_proc_entry"
    fi

    analyze_function_calls "call_usermodehelper"

    if [ -n "$MODULE_NAME" ]; then
        log " - Invisible Rootkit Module Name: $MODULE_NAME"
    else
        log " - Invisible Rootkit Module Name: Not found"
    fi

    if [ -n "$PROC_ENTRY" ]; then
        log " - Proc Entry: /proc/$PROC_ENTRY"
    else
        log " - Proc Entry: /proc/Not found"
    fi

    if [ -n "$BACKDOOR_PATH" ]; then
        log " - Backdoor Path: ${RED}$BACKDOOR_PATH${RESET}"
    else
        log " - Backdoor Path: ${RED}Not found${RESET}"
    fi

    extract_vermagic_info
}

#############################################
# main
#############################################
main() {
    log "${BLUE}============================================================${RESET}"
    log "${BLUE}      Rootkit Detection Scanner - Posix v.1.0-rev1${RESET}"
    log "${BLUE}============================================================${RESET}"

    need_root
    get_system_info
    log "${BLUE}============================================================${RESET}"

    scan_fs
    log ""
    log ""
    log "${BLUE}============================================================${RESET}"
    log "${BLUE}                        SCAN RESULT${RESET}"
    log "${BLUE}============================================================${RESET}"

    if [ "$HIDDEN_FOUND" -eq 0 ]; then
        log "${GREEN}[OK]${RESET} No Hidden Entry"
    else
        log "${RED}[Alert]${RESET} Hidden Entry Found!"
        if [ -n "$HIDDEN_LIST" ]; then
            tmp_h=`make_tmp rootkit_hsum`
            printf '%s\n' "$HIDDEN_LIST" >"$tmp_h"
            while IFS= read -r hidden; do
                [ -z "$hidden" ] && continue
                log "${RED}[!]${RESET} FilePath: ${RED}$hidden${RESET}"
                file_details "$hidden"
            done <"$tmp_h"
            rm -f "$tmp_h"
        fi
    fi

    if [ "$ROOTKIT_FOUND" -eq 0 ]; then
        log "${GREEN}[OK]${RESET} Rootkit Not Found"
    else
        log "${RED}[Alert]${RESET} Suspicious Rootkit Found!"
        if [ -n "$ROOTKIT_LIST" ]; then
            tmp_r=`make_tmp rootkit_rsum`
            printf '%s\n' "$ROOTKIT_LIST" >"$tmp_r"
            while IFS= read -r rootkit; do
                [ -z "$rootkit" ] && continue
                log "${RED}[!]${RESET} FilePath: ${RED}$rootkit${RESET}"
                file_details "$rootkit"
                ROOTKIT_NAME="$rootkit"
                log "${BLUE}-------------- Rootkit File Analysis -------------${RESET}"
                p_rootkit_info
                log "${BLUE}--------------------------------------------------${RESET}"
            done <"$tmp_r"
            rm -f "$tmp_r"
        fi
    fi

    if [ "$BACKDOOR_FOUND" -eq 0 ]; then
        log "${GREEN}[OK]${RESET} Backdoor Not Found"
    else
        log "${RED}[Alert]${RESET} Backdoor Found!"
        if [ -n "$BACKDOOR_LIST" ]; then
            tmp_b=`make_tmp rootkit_bsum`
            printf '%s\n' "$BACKDOOR_LIST" >"$tmp_b"
            while IFS= read -r backdoor; do
                [ -z "$backdoor" ] && continue
                log "${RED}[!]${RESET} FilePath: ${RED}$backdoor${RESET}"
                file_details "$backdoor"
            done <"$tmp_b"
            rm -f "$tmp_b"
        fi
    fi

    log "${BLUE}============================================================${RESET}"
    log "${BLUE}[*] Scan Complete!${RESET}"
    log "${BLUE}============================================================${RESET}"
}

if [ "$#" -gt 0 ]; then
    main "$@"
else
    main
fi

